/***************************************************************************************************
 * CodeTip: ICD-10-GM und OPS Code Dictionary
 * Version: 1.01
 * (c) 2011 Bernd Liebermann
 * E-Mail: software@bernd-liebermann.de
 * Web: http://www.bernd-liebermann.de
 *
 * Dieses Script darf kostenlos erweitert, verändert und in beliebige Webseiten eingebunden werden
 * unter der Voraussetzung, dass der Hinweis auf den Autor nicht entfernt wird.
 *
 **************************************************************************************************/

/**
 * Die CodeTip-Klasse.
 *
 * @param pattern
 * 		Das RegExp-Suchmuster für die nachzuschlagenden Codes.
 * 		Für vorgegebene Suchmusters kann auch "icd" bzw. "ops" angegeben werden.
 * @param lookupUrl
 * 		Die URL für das Nachschlage-Skript mit Platzhalter %q für	den nachzuschlagenden Kode.
 * @param wrapperClass
 * 		Optional. Der Name der CSS-Klasse, die auf die gefundenen Kodes
 *		angewendet werden soll. "code", wenn nichts anderes angegeben.
 * @param tooltipContainerId
 * 		Optional. Die Id des Elementes, das bei Überfahren mit der Maus
 *		ein- und ausgeblendet wird. "ct_container", wenn nicht angegeben.
 * @param tooltipContentId
 * 		Optional. Die Id des Kind-Elementes von tooltipContainerId,
 *	 	das die Beschreibung des Kodes enthält. Der Inhalt dieses Elementes
 *	 	wird bei der Anzeige des Tooltips komplett neu erstellt.
 * @param delay
 * 		Optional. Verzögerung bis zur Anzeige des Tooltips in Millisekunden.
 * 		Verhindert das ungewollte Anzeigen von Tooltips beim schnellen Überfahren
 * 		eines Kodes mit der Maus. Wenn nicht angegeben, wird 350ms als Standardwert
 * 		genommen.
 * @constructor
 *
 */

function CodeTip(pattern, lookupUrl, wrapperClass, tooltipContainerId, tooltipContentId, delay) {

	this.pattern = pattern;
	this.pattern = (this.pattern == "icd")
		? /([A-Z][0-9][0-9][^A-Za-z]\.?[0-9a-z-!*+]?[0-9a-z-!*+]?)/mg
		: this.pattern;
	this.pattern = (this.pattern == "ops")
		? /([135689]-[0-9][0-9][0-9abcdefghjkmnx]\.?[0-9a-z-!*+]?[0-9a-z-!*+]?)/mg
		: this.pattern;
	this.lookupUrl = lookupUrl;
	this.wrapperClass = (wrapperClass || "ct_code");
	this.tooltipContainerId = (tooltipContainerId || "ct_container");
	this.tooltipContentId = (tooltipContentId || "ct_content");
	this.delay = delay || 350;
	this.matchedTextNodes = [];
	this.matchSpanNodes = [];
	this.scanState = 0;

	var me = this;

	addEvent_ct(window, "load", function() {
		addEvent_ct(document.body, "click", function() {
			var el = document.getElementById(me.tooltipContainerId);
			if (el) {
				el.style.display = "none";
			}
		});
	});

	/**
	 * Steuert den zweistufigen Scan-Prozess (1=Scannen, 2=Fundstellen einkapseln)
	 * @param node DOM-Knoten, von dem aus gescannt werden soll. Normalerweise, aber nicht zwingend document.body.
	 */

	this.scan = function(node) {
		this.scanState = 1;
		this.matchedTextNodes = [];
		this.matchSpanNodes = [];
		this.traverse(node);
		this.scanState = 2;
		this.rebuildMatchedTextNodes();
		this.scanState = 3;
	};

	//

	/**
	 * Scannt unterhalb des übergebenen DOM-Knotens nach Codes, welche mit this.pattern
	 * übereinstimmen. Textknoten mit Fundstellen werden dem Feld matchedTextNodes hinzugefügt.
	 *
	 * @param node Knoten, unterhalb dem nach Codes gescannt werden soll.
	 */

	this.traverse = function (node) {
		if (node.nodeType == 3) {
			if (node.nodeValue.search(this.pattern) >= 0) {
				this.matchedTextNodes.push(node);
			}
		}
		var z = node.childNodes.length;
		for (var i = 0; i < z; i++) {
			if (node.childNodes[i].nodeName != "TEXTAREA") {
				this.traverse(node.childNodes[i]);
			}
		}
	};

	//

	/**
	 * Splittet die Textknoten mit Fundstellen in Textknoten und Span-Elementknoten auf.
	 * Die Span-Knoten mit den Fundstellen erhalten eine zufällig generierte id und werden
	 * im Feld this.matchSpanNodes gespeichert. Zusätzlich werden die Eventhandler onmouseover
	 * und onmouseout hier gesetzt.
	 */

	this.rebuildMatchedTextNodes = function() {
		var delimiter = guid();
		var n = this.matchedTextNodes.length;
		for (var i=0; i<n; i++) {
			var node = this.matchedTextNodes[i];
			var matches = node.nodeValue.match(this.pattern);
			if (matches) {
				var p = node.parentNode;
				var a = node.nodeValue.replace(this.pattern, delimiter).split(delimiter);
				var z = a.length - 1;
				var newNodes = [];
				for (var j=0; j<z; j++) {
					newNodes.push(document.createTextNode(a[j]));
					var span = document.createElement("span");
					span.className = this.wrapperClass;
					span.codeTip = this;
					span.id = guid();
					span.onmouseover = function() {
						this.codeTip.displayQueryResult(this);
					};
					span.onmouseout = this.hideQueryResult;
					var match = document.createTextNode(matches[j]);
					span.appendChild(match);
					newNodes.push(span);
					this.matchSpanNodes.push(span);
				}
				newNodes.push(document.createTextNode(a[a.length - 1]));
				p.replaceChild(newNodes[0], node);
				for (j=1; j<newNodes.length; j++) {
					if (newNodes[j-1].nextSibling) {
						p.insertBefore(newNodes[j], newNodes[j-1].nextSibling);
					} else {
						p.appendChild(newNodes[j]);
					}
				}
			}
		}
	};

	/**
	 * Fragt die Beschreibung zu dem Kode ab, der als nodeValue im übergebenen Knoten enthalten ist.
	 * Das Abfrageergebnis wird als Attribut query_result in Form eines JSON-String im
	 * übergebenen Knoten gespeichert.
	 *
	 * @param node Span-Knoten, der beim Scannen erzeugt wurde und einen gefundenen
	 * 		ICD- oder OPS-Code einschließt.
	 */

	this.queryCode = function(node) {
		var code = node.childNodes[0].nodeValue;
		var xhr = createXHR();
		if ((xhr) && (this.lookupUrl)) {
			xhr.target = node;
			xhr.open("GET", this.lookupUrl.replace("%q", code), true);
			xhr.onreadystatechange = function() {
				if (this.readyState == 4) {
					if (this.status == 200) {
						var r = eval("(" + this.responseText + ")");
						if (r.status == 0) {
							this.target.setAttribute("query_result", this.responseText);
							this.target.codeTip.displayQueryResult(this.target);
							this.target.style.cursor = "";
						}
					}
				}
			};
			// Wait Cursor soll nur zeitverzögert erscheinen, falls die Abfrage nicht innerhalb
			// einer halben Sekunde beantwortet wurde.
			window.setTimeout("showWaitCursor_ct('" + xhr.target.id + "')", 500);
			xhr.send(null);
		}
	};

	/**
	 * Zeigt die Beschreibung zum Kode als Tooltip an. Sollte die Beschreibung nicht vorliegen,
	 * wird stattdessen this.queryCode aufgerufen. Falls noch kein Element mit der Id
	 * this.tooltipContainerId vorhanden ist, werden tooltipContainer und tooltipContent
	 * durch Aufruf von this.createTooltipContainer erstellt.
	 *
	 * @param node Span-Knoten, der beim Scannen erzeugt wurde und einen gefundenen
	 * 		ICD- oder OPS-Code einschließt.
	 */

	this.displayQueryResult = function(node) {
		var r = node.getAttribute("query_result");
		if (!r) {
			this.queryCode(node);
		} else {
			var container =
				(document.getElementById(this.tooltipContainerId) || this.createTooltipContainer());
			if (container.style.display == "none") {
				var content = document.getElementById(this.tooltipContentId);
				r = eval("(" + r + ")");
				if (r.matches.length > 0) {
					while (content.childNodes.length > 0) {
						content.removeChild(content.childNodes[0]);
					}
					var tbl = document.createElement("table");
					var tbody = document.createElement("tbody");
					for (var i=0; i<r.matches.length; i++) {
						var tmp;
						var m = r.matches[i];
						var tr = document.createElement("tr");
						var td1 = document.createElement("td");
						td1.className = "ct_cell-code";
						var td2 = document.createElement("td");
						td2.className = "ct_cell-version";
						var td3 = document.createElement("td");
						td3.className = "ct_cell-text";
						if (i == 0) {
							tmp = document.createTextNode(m.code);
							td1.appendChild(tmp);
						}
						var v = (m.minVersion == m.maxVersion)
						 ? m.minVersion
						 : m.minVersion + "-" + m.maxVersion;
						tmp = document.createTextNode("[" + v + "]");
						td2.appendChild(tmp);
						tmp = document.createTextNode(m.text);
						td3.appendChild(tmp);
						tr.appendChild(td1);
						tr.appendChild(td2);
						tr.appendChild(td3);
						tbody.appendChild(tr);
						tbl.appendChild(tbody);
						content.appendChild(tbl);
					}
					var x = getX2(node);
					var y = getY(node);
					var w = getWindowWidth();
					container.style.top = (y + 10) + "px";
					if (x < (w - 300)) {
						container.style.left = (x + 10) + "px";
						container.style.right = "auto";
					} else {
						container.style.left = "auto";
						container.style.right = (w - getX(node) - 10) + "px";					
					}
					node.dataset.timeoutHandle = window.setTimeout(function() {
						container.style.display = "block";
					}, me.delay);

				}
			}
		}
	};

	/**
	 * Blendet die Beschreibung zum Kode wieder aus.
	 */

	this.hideQueryResult = function() {
		if (this.dataset.timeoutHandle) {
			window.clearTimeout(this.dataset.timeoutHandle);
		}
		var container = document.getElementById("ct_container");
		if (container) {
			container.style.display = "none";
		}
	};

	/**
	 * Erstellt tooltipContainer und tooltipContent.
	 *
	 * @return {Element}
	 */

	this.createTooltipContainer = function() {
		var container = document.createElement("div");
		container.id = this.tooltipContainerId;
		container.style.display = "none";
		var content = document.createElement("div");
		content.id = this.tooltipContentId;
		container.appendChild(content);
		document.body.appendChild(container);
		return container;
	};

	/**
	 * Erstellt eine zufällige ID in Form einer Pseuso-GUID.
	 * @return {String} Pseudo-GUID.
	 */

	function guid() {
		return (s4()+s4()+"-"+s4()+"-"+s4()+"-"+s4()+"-"+s4()+s4()+s4());
		function s4() {
			return (((1+Math.random())*0x10000)|0).toString(16).substring(1);
		}
	}

	/**
	 * Gibt die horizontale Position der linken Kante des übergebenen Elementes zurück.
	 * Wird zur Positionierung der Tooltip-Anzeige benötigt.
	 *
	 * @param el Element, für das die Koordinate bestimmt werden soll.
	 * @return {Number} Linke X-Koordinate.
	 */

	function getX(el) {
		var x = 0;
		while (el) {
			x += el.offsetLeft;
			el = el.offsetParent;
		}
		return x;
	}

	/**
	 * Gibt die horizontale Position der rechten Kante des übergebenen Elementes zurück.
	 * Wird zur Positionierung der Tooltip-Anzeige benötigt.
	 *
	 * @param el Element, für das die Koordinate bestimmt werden soll.
	 * @return {Number} Rechte X-Koordinate.
	 */
	function getX2(el) {
		var x2 = getX(el) + el.offsetWidth;
		return x2;
	}

	/**
	 * Gibt die vertikale Position der oberen Kante des übergebenen Elementes zurück.
	 * Wird zur Positionierung der Tooltip-Anzeige benötigt.
	 *
	 * @param el Element, für das die Koordinate bestimmt werden soll.
	 * @return {Number} Obere Y-Koordinate.
	 */

	function getY(el) {
		var y = 0;
		while (el) {
			y += el.offsetTop;
			el = el.offsetParent;
		}
		return y;
	}

	/**
	 * Gibt die vertikale Position der unteren Kante des übergebenen Elementes zurück.
	 * Wird zur Positionierung der Tooltip-Anzeige benötigt.
	 *
	 * @param el Element, für das die Koordinate bestimmt werden soll.
	 * @return {Number} Untere Y-Koordinate.
	 */

	function getY2(el) {
		var y2 = getY(el) + el.offsetHeight;
		return y2;
	}

	/**
	 * Gibt die Breite des Browser-Fensters zurück.
	 * Wird zur Positionierung der Tooltip-Anzeige benötigt.
	 *
	 * @return {Number} Breite des Browser-Fensters in Pixel.
	 */

	function getWindowWidth() {
		var winW = 1000;
		if (document.body && document.body.offsetWidth) {
		 winW = document.body.offsetWidth;
		}
		if (document.compatMode=='CSS1Compat' &&
			document.documentElement &&
			document.documentElement.offsetWidth ) {
		 winW = document.documentElement.offsetWidth;
		}
		if (window.innerWidth && window.innerHeight) {
		 winW = window.innerWidth;
		}
		return winW;
	}

	/**
	 * Gibt die Höhe des Browser-Fensters zurück.
	 * Wird zur Positionierung der Tooltip-Anzeige benötigt.
	 *
	 * @return {Number} Höhe des Browser-Fensters in Pixel.
	 */

	function getWindowHeight() {
		var winH = 500;
		if (document.body && document.body.offsetWidth) {
		 winH = document.body.offsetHeight;
		}
		if (document.compatMode=='CSS1Compat' &&
			document.documentElement &&
			document.documentElement.offsetWidth ) {
		 winH = document.documentElement.offsetHeight;
		}
		if (window.innerWidth && window.innerHeight) {
		 winH = window.innerHeight;
		}
		return winH;
	}

	//
	/**
	 * Erstellt ein neues XMLHttpRequest-Objekt.
	 *
	 * @return {*} XMLHttpRequest-Objekt.
	 */
	function createXHR() {
		var xmlHttp = null;
		try {
			xmlHttp = new XMLHttpRequest();
		} catch(e) {
			try {
				xmlHttp  = new ActiveXObject("Microsoft.XMLHTTP");
			} catch(e) {
				try {
					xmlHttp  = new ActiveXObject("Msxml2.XMLHTTP");
				} catch(e) {
					xmlHttp  = null;
				}
			}
		}
		return xmlHttp;
	}
}

/**
 * Stellt die Cursor-Formatierung des Elementes mit der übergebenen Id auf "wait" ein,
 * falls noch kein query_result eingetroffen ist.
 *
 * @param id Id des Elements, dessen Cursor geändert werden soll.
 */

function showWaitCursor_ct(id) {

	var span = document.getElementById(id);
	if (span && !span.getAttribute("query_result")) {
		span.style.cursor = "wait";
	}

}

/**
 * Der folgende Kode geht auf Dean Edwards zurück. Er erlaubt es mittels eines einzigen
 * Funktionsaufrufes Ereignisbehandlungsroutinen sowohl in modernen Browsern als auch
 * im IE 6 zu registrieren.
 *
 * @param element DOM-Element, für das der Ereignishandler registriert werden soll.
 * @param type Ereignis (ohne vorangestelltes "on")
 * @param handler Ereignishandler.
 */

// written by Dean Edwards, 2005
// with input from Tino Zijdel, Matthias Miller, Diego Perini

// http://dean.edwards.name/weblog/2005/10/add-event/

function addEvent_ct(element, type, handler) {
	if (element.addEventListener) {
		element.addEventListener(type, handler, false);
	} else {
		if (!handler.$$guid) handler.$$guid = addEvent_ct.guid++;
		if (!element.events) element.events = {};
		var handlers = element.events[type];
		if (!handlers) {
			handlers = element.events[type] = {};
			if (element["on" + type]) {
				handlers[0] = element["on" + type];
			}
		}
		handlers[handler.$$guid] = handler;
		element["on" + type] = handleEvent_ct;
	}
}

addEvent_ct.guid = 1;

function removeEvent_ct(element, type, handler) {
	if (element.removeEventListener) {
		element.removeEventListener(type, handler, false);
	} else {
		if (element.events && element.events[type]) {
			delete element.events[type][handler.$$guid];
		}
	}
}

function handleEvent_ct(event) {
	var returnValue = true;
	event = event || fixEvent_ct(((this.ownerDocument || this.document || this).parentWindow || window).event);
	var handlers = this.events[event.type];
	for (var i in handlers) {
		this.$$handleEvent_ct = handlers[i];
		if (this.$$handleEvent_ct(event) === false) {
			returnValue = false;
		}
	}
	return returnValue;
}

function fixEvent_ct(event) {
	event.preventDefault = fixEvent_ct.preventDefault;
	event.stopPropagation = fixEvent_ct.stopPropagation;
	return event;
}

fixEvent_ct.preventDefault = function() {
	this.returnValue = false;
};

fixEvent_ct.stopPropagation = function() {
	this.cancelBubble = true;
};
